---------------------------------------------------------------------
-- Two-Dimensional Interval Packing Challenge
--  Itzik Ben-Gan
---------------------------------------------------------------------

---------------------------------------------------------------------
-- The Challenge
---------------------------------------------------------------------

---------------------------------------------------------------------
-- DDL and Sample Data
---------------------------------------------------------------------

SET NOCOUNT ON;
USE tempdb;

DROP TABLE IF EXISTS dbo.Schedule;

CREATE TABLE dbo.Schedule
(
  id         INT     NOT NULL CONSTRAINT PK_Schedule PRIMARY KEY,
  student    CHAR(3) NOT NULL,
  fromdate   INT     NOT NULL,
  todate     INT     NOT NULL,
  fromperiod INT     NOT NULL,
  toperiod   INT     NOT NULL
);

INSERT INTO dbo.Schedule(id, student, fromdate, todate, fromperiod, toperiod) VALUES
    (1, 'Amy',  1,  7,  7,  9),
    (2, 'Amy',  3,  9,  5,  8), 
    (3, 'Amy', 10, 12,  1,  3), 
    (4, 'Ted',  1,  5, 11, 14),
    (5, 'Ted',  7, 11, 13, 16);

-- Show current unnormalized schedule graphically
-- Depict whole/inclusive nature of each todate and toperiod
-- Figure 1, Unnormalized Schedule
SELECT student,
  GEOMETRY::STGeomFromText('GEOMETRYCOLLECTION('
    + STRING_AGG(CONCAT('POLYGON((', fromdate  , ' ', fromperiod  , ',', 
                                     fromdate  , ' ', toperiod + 1, ',', 
                                     todate + 1, ' ', toperiod + 1, ',', 
                                     todate + 1, ' ', fromperiod  , ',', 
                                     fromdate  , ' ', fromperiod  , '))'), ',') 
    + ')', 0) AS shape
FROM dbo.Schedule
GROUP BY student;

-- Identify Amy's schedule that contains date 4, period 8
-- Get more than one match back
SELECT id, student, fromdate, todate, fromperiod, toperiod
FROM dbo.Schedule
WHERE student = 'Amy'
  AND 4 BETWEEN fromdate AND todate
  AND 8 BETWEEN fromperiod AND toperiod;

id          student fromdate    todate      fromperiod  toperiod
----------- ------- ----------- ----------- ----------- -----------
1           Amy     1           7           7           9
2           Amy     3           9           5           8

-- Desired output
student fromdate    todate      fromperiod  toperiod
------- ----------- ----------- ----------- -----------
Amy     1           2           7           9
Amy     3           7           5           9
Amy     8           9           5           8
Amy     10          12          1           3
Ted     1           5           11          14
Ted     7           11          13          16

-- Figure 2, Desired Normalized Schedule

---------------------------------------------------------------------
-- Solution for One Dimensional Packing
---------------------------------------------------------------------

-- Suppose that you only had period ranges and needed to normalize them
SELECT id, student, fromperiod, toperiod
FROM dbo.Schedule
ORDER BY student, fromperiod, toperiod;

-- Output
id          student fromperiod  toperiod
----------- ------- ----------- -----------
3           Amy     1           3
2           Amy     5           8
1           Amy     7           9
4           Ted     11          14
5           Ted     13          16

-- Desired output
student fromperiod  toperiod
------- ----------- -----------
Amy     1           3
Amy     5           9
Ted     11          16

-- Step 1, unpack
SELECT S.id, S.student, P.value AS p
FROM dbo.Schedule AS S
  CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
ORDER BY S.student, p, S.id;

id          student p
----------- ------- -----------
3           Amy     1
3           Amy     2
3           Amy     3
2           Amy     5
2           Amy     6
1           Amy     7
2           Amy     7
1           Amy     8
2           Amy     8
1           Amy     9
4           Ted     11
4           Ted     12
4           Ted     13
5           Ted     13
4           Ted     14
5           Ted     14
5           Ted     15
5           Ted     16

-- Step 2, compute a group identifier
SELECT S.id, S.student, P.value AS p,
  DENSE_RANK() OVER(PARTITION BY S.student ORDER BY P.value) AS drk,
  P.value - DENSE_RANK() OVER(PARTITION BY S.student ORDER BY P.value) AS grp_p
FROM dbo.Schedule AS S
  CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
ORDER BY S.student, p, S.id;

id          student p           drk                  grp_p
----------- ------- ----------- -------------------- --------------------
3           Amy     1           1                    0
3           Amy     2           2                    0
3           Amy     3           3                    0
2           Amy     5           4                    1
2           Amy     6           5                    1
1           Amy     7           6                    1
2           Amy     7           6                    1
1           Amy     8           7                    1
2           Amy     8           7                    1
1           Amy     9           8                    1
4           Ted     11          1                    10
4           Ted     12          2                    10
4           Ted     13          3                    10
5           Ted     13          3                    10
4           Ted     14          4                    10
5           Ted     14          4                    10
5           Ted     15          5                    10
5           Ted     16          6                    10

-- Step 3, group by group identifier and compute min/max p values as range delimiters
WITH C AS
(
  SELECT S.id, S.student, P.value AS p,
    DENSE_RANK() OVER(PARTITION BY S.student ORDER BY P.value) AS drk,
    P.value - DENSE_RANK() OVER(PARTITION BY S.student ORDER BY P.value) AS grp_p
  FROM dbo.Schedule AS S
    CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
)
SELECT student, MIN(p) AS fromperiod, MAX(p) AS toperiod
FROM C
GROUP BY student, grp_p
ORDER BY student, fromperiod;

student fromperiod  toperiod
------- ----------- -----------
Amy     1           3
Amy     5           9
Ted     11          16

---------------------------------------------------------------------
-- Solution for Two-Dimensional Packing
---------------------------------------------------------------------

-- Solution Step1, Pixelate
-- Unpack schedule to individual date (d), period (p) values
SELECT S.id, S.student, D.value AS d, P.value AS p
FROM dbo.Schedule AS S
  CROSS APPLY GENERATE_SERIES(S.fromdate, S.todate) AS D
  CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
ORDER BY S.student, d, p, S.id;

id          student d           p
----------- ------- ----------- -----------
1           Amy     1           7
1           Amy     1           8
1           Amy     1           9
1           Amy     2           7
1           Amy     2           8
1           Amy     2           9
2           Amy     3           5
2           Amy     3           6
1           Amy     3           7
2           Amy     3           7
1           Amy     3           8
2           Amy     3           8
1           Amy     3           9
2           Amy     4           5
2           Amy     4           6
1           Amy     4           7
2           Amy     4           7
1           Amy     4           8
2           Amy     4           8
1           Amy     4           9
2           Amy     5           5
2           Amy     5           6
1           Amy     5           7
2           Amy     5           7
1           Amy     5           8
2           Amy     5           8
1           Amy     5           9
2           Amy     6           5
2           Amy     6           6
1           Amy     6           7
2           Amy     6           7
1           Amy     6           8
2           Amy     6           8
1           Amy     6           9
2           Amy     7           5
2           Amy     7           6
1           Amy     7           7
2           Amy     7           7
1           Amy     7           8
2           Amy     7           8
1           Amy     7           9
2           Amy     8           5
2           Amy     8           6
2           Amy     8           7
2           Amy     8           8
2           Amy     9           5
2           Amy     9           6
2           Amy     9           7
2           Amy     9           8
3           Amy     10          1
3           Amy     10          2
3           Amy     10          3
3           Amy     11          1
3           Amy     11          2
3           Amy     11          3
3           Amy     12          1
3           Amy     12          2
3           Amy     12          3
...

-- Depict Step 1's result grpahically
-- Figure 3, Pixelated Schedule
WITH Pixels AS
(
  SELECT S.id, S.student, D.value AS d, P.value AS p
  FROM dbo.Schedule AS S
    CROSS APPLY GENERATE_SERIES(S.fromdate, S.todate) AS D
    CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
)
SELECT student,
  GEOMETRY::STGeomFromText('GEOMETRYCOLLECTION('
    + STRING_AGG(CONCAT('POLYGON((', d    , ' ', p    , ',', 
                                     d    , ' ', p + 1, ',', 
                                     d + 1, ' ', p + 1, ',', 
                                     d + 1, ' ', p    , ',', 
                                     d    , ' ', p    , '))'), ',') 
    + ')', 0) AS shape
FROM Pixels
GROUP BY student;

-- Step 2, Pack period ranges per student and date
-- Achieved by assigning group identifier grp_p to each distinct range of consecutive p values in student and d group
WITH PixelsAndPeriodGroupIDs AS
(
  SELECT S.id, S.student, D.value AS d, P.value AS p,
    P.value - DENSE_RANK() OVER(PARTITION BY S.student, D.value ORDER BY P.value) AS grp_p
  FROM dbo.Schedule AS S
    CROSS APPLY GENERATE_SERIES(S.fromdate, S.todate) AS D
    CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
)
SELECT student, d, MIN(p) AS fromperiod, MAX(p) AS toperiod
FROM PixelsAndPeriodGroupIDs
GROUP BY student, d, grp_p
ORDER BY student, d, grp_p;

-- Output of CTE PixelsAndPeriodGroupIDs inner query
id          student d           p           grp_p
----------- ------- ----------- ----------- --------------------
1           Amy     1           7           6
1           Amy     1           8           6
1           Amy     1           9           6
1           Amy     2           7           6
1           Amy     2           8           6
1           Amy     2           9           6
2           Amy     3           5           4
2           Amy     3           6           4
2           Amy     3           7           4
1           Amy     3           7           4
1           Amy     3           8           4
2           Amy     3           8           4
1           Amy     3           9           4
2           Amy     4           5           4
2           Amy     4           6           4
2           Amy     4           7           4
1           Amy     4           7           4
1           Amy     4           8           4
2           Amy     4           8           4
1           Amy     4           9           4
2           Amy     5           5           4
2           Amy     5           6           4
2           Amy     5           7           4
1           Amy     5           7           4
1           Amy     5           8           4
2           Amy     5           8           4
1           Amy     5           9           4
2           Amy     6           5           4
2           Amy     6           6           4
2           Amy     6           7           4
1           Amy     6           7           4
1           Amy     6           8           4
2           Amy     6           8           4
1           Amy     6           9           4
2           Amy     7           5           4
2           Amy     7           6           4
2           Amy     7           7           4
1           Amy     7           7           4
1           Amy     7           8           4
2           Amy     7           8           4
1           Amy     7           9           4
2           Amy     8           5           4
2           Amy     8           6           4
2           Amy     8           7           4
2           Amy     8           8           4
2           Amy     9           5           4
2           Amy     9           6           4
2           Amy     9           7           4
2           Amy     9           8           4
3           Amy     10          1           0
3           Amy     10          2           0
3           Amy     10          3           0
3           Amy     11          1           0
3           Amy     11          2           0
3           Amy     11          3           0
3           Amy     12          1           0
3           Amy     12          2           0
3           Amy     12          3           0
...

-- Output of outer query against CTE

student d           fromperiod  toperiod
------- ----------- ----------- -----------
Amy     1           7           9
Amy     2           7           9
Amy     3           5           9
Amy     4           5           9
Amy     5           5           9
Amy     6           5           9
Amy     7           5           9
Amy     8           5           8
Amy     9           5           8
Amy     10          1           3
Amy     11          1           3
Amy     12          1           3
Ted     1           11          14
Ted     2           11          14
Ted     3           11          14
Ted     4           11          14
Ted     5           11          14
Ted     7           13          16
Ted     8           13          16
Ted     9           13          16
Ted     10          13          16
Ted     11          13          16

-- Depict Step 2's result graphically
-- Figure 4, Daily Period Ranges
WITH PixelsAndPeriodGroupIDs AS
(
  SELECT S.id, S.student, D.value AS d, P.value AS p,
    P.value - DENSE_RANK() OVER(PARTITION BY S.student, D.value ORDER BY P.value) AS grp_p
  FROM dbo.Schedule AS S
    CROSS APPLY GENERATE_SERIES(S.fromdate, S.todate) AS D
    CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
),
DailyPeriodRanges AS
(
  SELECT student, d, MIN(p) AS fromperiod, MAX(p) AS toperiod
  FROM PixelsAndPeriodGroupIDs
  GROUP BY student, d, grp_p
)
SELECT student,
  GEOMETRY::STGeomFromText('GEOMETRYCOLLECTION('
    + STRING_AGG(CONCAT('POLYGON((', d    , ' ', fromperiod  , ',', 
                                     d    , ' ', toperiod + 1, ',', 
                                     d + 1, ' ', toperiod + 1, ',', 
                                     d + 1, ' ', fromperiod  , ',', 
                                     d    , ' ', fromperiod  , '))'), ',') 
    + ')', 0) AS shape
FROM DailyPeriodRanges
GROUP BY student;

-- Step 3, Pack date ranges per period group
WITH PixelsAndPeriodGroupIDs AS
(
  SELECT S.id, S.student, D.value AS d, P.value AS p,
    P.value - DENSE_RANK() OVER(PARTITION BY S.student, D.value ORDER BY P.value) AS grp_p
  FROM dbo.Schedule AS S
    CROSS APPLY GENERATE_SERIES(S.fromdate, S.todate) AS D
    CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
),
PeriodRangesAndDateGroupIDs AS
(
  SELECT student, d, MIN(p) AS fromperiod, MAX(p) AS toperiod,
    d - DENSE_RANK() OVER(PARTITION BY student, MIN(p), MAX(p) ORDER BY d) as grp_d
  FROM PixelsAndPeriodGroupIDs
  GROUP BY student, d, grp_p
)
SELECT student, MIN(d) AS fromdate, MAX(d) AS todate, fromperiod, toperiod
FROM PeriodRangesAndDateGroupIDs
GROUP BY student, fromperiod, toperiod, grp_d
ORDER BY student, fromdate, fromperiod;

-- Output of CTE PeriodRangesAndDateGroupIDs inner query
student d           fromperiod  toperiod    grp_d
------- ----------- ----------- ----------- --------------------
Amy     10          1           3           9
Amy     11          1           3           9
Amy     12          1           3           9
Amy     8           5           8           7
Amy     9           5           8           7
Amy     3           5           9           2
Amy     4           5           9           2
Amy     5           5           9           2
Amy     6           5           9           2
Amy     7           5           9           2
Amy     1           7           9           0
Amy     2           7           9           0
Ted     1           11          14          0
Ted     2           11          14          0
Ted     3           11          14          0
Ted     4           11          14          0
Ted     5           11          14          0
Ted     7           13          16          6
Ted     8           13          16          6
Ted     9           13          16          6
Ted     10          13          16          6
Ted     11          13          16          6

-- Output of outer query, showing desired normalized schedule
student fromdate    todate      fromperiod  toperiod
------- ----------- ----------- ----------- -----------
Amy     1           2           7           9
Amy     3           7           5           9
Amy     8           9           5           8
Amy     10          12          1           3
Ted     1           5           11          14
Ted     7           11          13          16

-- Show solution areas graphically
-- Figure 2, Normalized Schedule
WITH PixelsAndPeriodGroupIDs AS
(
  SELECT S.id, S.student, D.value AS d, P.value AS p,
    P.value - DENSE_RANK() OVER(PARTITION BY S.student, D.value ORDER BY P.value) AS grp_p
  FROM dbo.Schedule AS S
    CROSS APPLY GENERATE_SERIES(S.fromdate, S.todate) AS D
    CROSS APPLY GENERATE_SERIES(S.fromperiod, S.toperiod) AS P
),
PeriodRangesAndDateGroupIDs AS
(
  SELECT student, d, MIN(p) AS fromperiod, MAX(p) AS toperiod,
    d - DENSE_RANK() OVER(PARTITION BY student, MIN(p), MAX(p) ORDER BY d) as grp_d
  FROM PixelsAndPeriodGroupIDs
  GROUP BY student, d, grp_p
),
NormalizedSchedule AS
(
  SELECT student, MIN(d) AS fromdate, MAX(d) AS todate, fromperiod, toperiod
  FROM PeriodRangesAndDateGroupIDs
  GROUP BY student, fromperiod, toperiod, grp_d
)
SELECT student,
  GEOMETRY::STGeomFromText('GEOMETRYCOLLECTION('
    + STRING_AGG(CONCAT('POLYGON((', fromdate  , ' ', fromperiod  , ',', 
                                     fromdate  , ' ', toperiod + 1, ',', 
                                     todate + 1, ' ', toperiod + 1, ',', 
                                     todate + 1, ' ', fromperiod  , ',', 
                                     fromdate  , ' ', fromperiod  , '))'), ',') 
    + ')', 0) AS shape
FROM NormalizedSchedule
GROUP BY student;